<?php
// Include the admin header to add sidebar and top navigation
include 'admin_header.php'; 

// Include database connection
include '../config/database.php';

// Fetch the total number of registered users (students)
$stmt = $pdo->prepare('SELECT COUNT(*) AS total_users FROM users');
$stmt->execute();
$totalUsersResult = $stmt->fetch(PDO::FETCH_ASSOC);
$totalUsers = $totalUsersResult['total_users'];

// Fetch the total number of valid index numbers
$stmt = $pdo->prepare('SELECT COUNT(*) AS total_valid_index FROM valid_index_numbers');
$stmt->execute();
$totalValidIndexResult = $stmt->fetch(PDO::FETCH_ASSOC);
$totalValidIndex = $totalValidIndexResult['total_valid_index'];

// Fetch the total number of registered lecturers
$stmt = $pdo->prepare('SELECT COUNT(*) AS total_lecturers FROM lecturer_login');
$stmt->execute();
$totalLecturersResult = $stmt->fetch(PDO::FETCH_ASSOC);
$totalLecturers = $totalLecturersResult['total_lecturers'];

// Fetch the total number of valid staff IDs
$stmt = $pdo->prepare('SELECT COUNT(*) AS total_valid_staff FROM valid_staff_id');
$stmt->execute();
$totalValidStaffResult = $stmt->fetch(PDO::FETCH_ASSOC);
$totalValidStaff = $totalValidStaffResult['total_valid_staff'];

// Fetch the organization and district with the most interns
$stmt = $pdo->prepare('
    SELECT organisation_name, district, COUNT(*) AS intern_count
    FROM student_details
    GROUP BY organisation_name, district
    ORDER BY intern_count DESC
    LIMIT 1
');
$stmt->execute();
$mostInternsResult = $stmt->fetch(PDO::FETCH_ASSOC);
$topOrganization = $mostInternsResult['organisation_name'] ?? 'N/A';
$topDistrict = $mostInternsResult['district'] ?? 'N/A';
$internCount = $mostInternsResult['intern_count'] ?? 0;

// Fetch the number of unique organizations
$stmt = $pdo->prepare('SELECT COUNT(DISTINCT organisation_name) AS unique_organizations FROM student_details');
$stmt->execute();
$uniqueOrganizationsResult = $stmt->fetch(PDO::FETCH_ASSOC);
$uniqueOrganizations = $uniqueOrganizationsResult['unique_organizations'] ?? 0;

// Fetch student distribution by region and group (Industrial Internship and Teaching Practice)
$query = '
    SELECT sd.region, u.group AS student_group, COUNT(*) AS student_count
    FROM student_details sd
    JOIN users u ON sd.user_id = u.id
    WHERE sd.status = "submitted"
    GROUP BY sd.region, u.group
';
$stmt = $pdo->prepare($query);
$stmt->execute();
$studentDistribution = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch number of students by group
$stmt = $pdo->prepare('SELECT `group`, COUNT(*) AS group_count FROM users GROUP BY `group`');
$stmt->execute();
$groupDistribution = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Prepare data for the region chart with initials
$regions = [];
$tpCounts = [];
$indCounts = [];
$allCounts = [];
$regionInitials = [
    'Central Region' => 'CR',
    'Greater Accra Region' => 'GAR',
    'Western North' => 'WN',
    'Upper West Region' => 'UWR',
    'Ashanti Region' => 'ASH',
    'Bono East Region' => 'BER',
    'BONO' => 'BR',
    'Eastern Region' => 'ER',
    'Northern Region' => 'NR',
    'Oti Region' => 'OR',
    'Savannah Region' => 'SR',
    'Upper East Region' => 'UER',
    'Volta Region' => 'VR',
    'Ahafo Region' => 'AR',
    'Western Region' => 'WR',
    'North East' => 'NE',
];

// Extract regions and initialize counts
foreach ($studentDistribution as $data) {
    $regionInitial = $regionInitials[$data['region']] ?? $data['region'];  // Map full region names to initials
    if (!in_array($regionInitial, $regions)) {
        $regions[] = $regionInitial;  // Add unique region initials to the list
    }
}

// Initialize arrays with 0 values for each region
foreach ($regions as $region) {
    $tpCounts[$region] = 0;
    $indCounts[$region] = 0;
    $allCounts[$region] = 0;
}

// Populate counts for each group and region
foreach ($studentDistribution as $data) {
    $regionInitial = $regionInitials[$data['region']] ?? $data['region'];  // Map full region names to initials
    if ($data['student_group'] == 'Teaching Practice') {
        $tpCounts[$regionInitial] = $data['student_count'];
    } elseif ($data['student_group'] == 'Industrial Internship') {
        $indCounts[$regionInitial] = $data['student_count'];
    }
    $allCounts[$regionInitial] = $tpCounts[$regionInitial] + $indCounts[$regionInitial];
}

// Prepare data for group donut chart
$groups = [];
$groupCounts = [];
foreach ($groupDistribution as $row) {
    $groups[] = $row['group'];
    $groupCounts[] = $row['group_count'];
}

// Fetch recent statuses and include the previous organization from the former_organisations table
$stmt = $pdo->prepare('
    SELECT tr.created_at, u.index_number, sd.first_name, sd.surname, fo.organisation_name AS previous_organisation, 
           tr.new_organisation_name, tr.reason, tr.status
    FROM transfer_requests tr 
    JOIN users u ON tr.user_id = u.id 
    JOIN student_details sd ON u.id = sd.user_id
    LEFT JOIN former_organisations fo ON tr.id = fo.transfer_id
    ORDER BY tr.created_at DESC
    LIMIT 4
');
$stmt->execute();
$recentStatus = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SOB-IMS</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome for icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <!-- Chart.js for charts -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <!-- Raleway Font -->
    <link href="https://fonts.googleapis.com/css2?family=Raleway:wght@400;600&display=swap" rel="stylesheet">
    <!-- Custom CSS -->
    <link href="admin_style.css" rel="stylesheet">
    <style>
        body {
            font-family: 'Raleway', sans-serif;
        }

        .sidebar-heading, .navbar-brand {
            font-family: 'Raleway', sans-serif;
            font-weight: 600;
        }

        h1 {
            font-family: 'Raleway', sans-serif;
            font-weight: 600;
        }

        .card-summary {
            height: 180px;
            display: flex;
            flex-direction: column;
            justify-content: center;
        }

        .card-title {
            font-size: 1rem;
            display: flex;
            align-items: center;
            justify-content: start;
        }

        .card-title i {
            margin-right: 8px;
        }

        .card-text {
            font-size: 2rem;
            margin-top: 10px;
            text-align: left;
        }

        .card-body p {
            text-align: left;
            margin-top: 5px;
        }

        .top-org-text {
            margin-bottom: 0;
        }

        .btn-primary {
            background-color: #4084f3 !important; /* Primary color */
            border-color: #4084f3 !important;
        }

        .btn-secondary {
            background-color: #32aa53 !important; /* Secondary color */
            border-color: #32aa53 !important;
        }

        .btn-warning {
            background-color: #fdbb05 !important; /* Warning color */
            border-color: #fdbb05 !important;
        }

        .btn-danger {
            background-color: #e94335 !important; /* Danger color */
            border-color: #e94335 !important;
        }

        .bg-primary {
            background-color: #4084f3 !important;
        }

        .bg-secondary {
            background-color: #32aa53 !important;
        }

        .bg-warning {
            background-color: #fdbb05 !important;
        }

        .bg-danger {
            background-color: #e94335 !important;
        }
    </style>
</head>
<body>
    <!-- Main Content -->
    <div class="container-fluid">
        <h1 class="mt-4">SOB-IMS</h1>
        <p>Welcome to the SOB-IMS Dashboard. Use the sidebar to navigate through the different sections.</p>

        <!-- Summary Cards -->
        <div class="row">
            <!-- Total Users Card -->
            <div class="col-lg-3 col-md-6">
                <div class="card bg-primary text-white mb-3 card-summary">
                    <div class="card-body">
                        <div class="card-title"><i class="fas fa-users"></i> Total Users</div>
                        <h2 class="card-text"><?php echo $totalUsers; ?> / <?php echo $totalValidIndex; ?></h2>
                    </div>
                </div>
            </div>

            <!-- Total Lecturers Card -->
            <div class="col-lg-3 col-md-6">
                <div class="card bg-secondary text-white mb-3 card-summary">
                    <div class="card-body">
                        <div class="card-title"><i class="fas fa-chalkboard-teacher"></i> Total Lecturers</div>
                        <h2 class="card-text"><?php echo $totalLecturers; ?> / <?php echo $totalValidStaff; ?></h2>
                    </div>
                </div>
            </div>

            <!-- Top Organization Card -->
            <div class="col-lg-3 col-md-6">
                <div class="card bg-warning text-dark mb-3 card-summary">
                    <div class="card-body">
                        <div class="card-title"><i class="fas fa-building"></i> Top Organization</div>
                        <h5 class="top-org-text"><?php echo $topOrganization; ?></h5>
                        <p class="top-org-text"><?php echo $topDistrict; ?> District</p>
                        <p class="top-org-text">Interns: <?php echo $internCount; ?></p>
                    </div>
                </div>
            </div>

            <!-- Unique Organizations Card -->
            <div class="col-lg-3 col-md-6">
                <div class="card bg-danger text-white mb-3 card-summary">
                    <div class="card-body">
                        <div class="card-title"><i class="fas fa-university"></i> Unique Organizations</div>
                        <h2 class="card-text"><?php echo $uniqueOrganizations; ?></h2>
                    </div>
                </div>
            </div>
        </div>

        <!-- Charts Row -->
        <div class="row">
            <!-- Student Distribution by Region Chart -->
            <div class="col-lg-6">
                <div class="card mb-4">
                    <div class="card-header">
                        Student Distribution by Region
                    </div>
                    <div class="card-body">
                        <canvas id="studentRegionChart"></canvas>
                    </div>
                </div>
            </div>

            <!-- Student Group Distribution Chart (Doughnut) -->
            <div class="col-lg-6">
                <div class="card mb-4">
                    <div class="card-header">
                        Number of Students by Group
                    </div>
                    <div class="card-body">
                        <canvas id="groupDistributionChart"></canvas>
                    </div>
                </div>
            </div>
        </div>

        <!-- Recent Status Table -->
        <div class="card mb-4">
            <div class="card-header d-flex justify-content-between align-items-center">
                Recent Status
                <a href="transfers.php" class="btn btn-primary btn-sm">View All Transfers</a>
            </div>
            <div class="card-body">
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th scope="col">Date</th>
                            <th scope="col">Student ID</th>
                            <th scope="col">Student Name</th>
                            <th scope="col">Previous Organization</th>
                            <th scope="col">New Organization</th>
                            <th scope="col">Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recentStatus as $status): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($status['created_at']); ?></td>
                                <td><?php echo htmlspecialchars($status['index_number']); ?></td>
                                <td><?php echo htmlspecialchars($status['first_name'] . ' ' . $status['surname']); ?></td>
                                <td><?php echo htmlspecialchars($status['previous_organisation']); ?></td>
                                <td><?php echo htmlspecialchars($status['new_organisation_name']); ?></td>
                                <td>
                                    <span class="badge bg-<?php echo $status['status'] == 'approved' ? 'success' : ($status['status'] == 'rejected' ? 'danger' : 'warning text-dark'); ?>">
                                        <?php echo ucfirst($status['status']); ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Modal to Display Students -->
    <div class="modal fade" id="studentModal" tabindex="-1" aria-labelledby="studentModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="studentModalLabel">Students from <span id="region-name"></span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Name</th>
                                <th>District</th>
                            </tr>
                        </thead>
                        <tbody id="student-table-body">
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Bootstrap Bundle with Popper -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <!-- Chart.js -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

    <script>
        const regions = <?php echo json_encode($regions); ?>;
        const tpCounts = <?php echo json_encode(array_values($tpCounts)); ?>;
        const indCounts = <?php echo json_encode(array_values($indCounts)); ?>;
        const totalCounts = <?php echo json_encode(array_values($allCounts)); ?>;
        const groups = <?php echo json_encode($groups); ?>;
        const groupCounts = <?php echo json_encode($groupCounts); ?>;

        // Line chart for Student Distribution by Region
        const ctx1 = document.getElementById('studentRegionChart').getContext('2d');
        const studentRegionChart = new Chart(ctx1, {
            type: 'line',
            data: {
                labels: regions,
                datasets: [{
                    label: 'Teaching Practice',
                    data: tpCounts,
                    borderColor: 'rgba(255, 99, 132, 1)',
                    borderWidth: 2,
                    fill: false
                }, {
                    label: 'Industrial Internship',
                    data: indCounts,
                    borderColor: 'rgba(54, 162, 235, 1)',
                    borderWidth: 2,
                    fill: false
                }, {
                    label: 'Total',
                    data: totalCounts,
                    borderColor: 'rgba(75, 192, 192, 1)',
                    borderWidth: 2,
                    fill: false,
                    borderDash: [5, 5] // Dashed line for total
                }]
            },
            options: {
                onClick: function(evt, activeElements) {
                    if (activeElements.length > 0) {
                        const datasetIndex = activeElements[0].datasetIndex;
                        const dataIndex = activeElements[0].index;
                        const selectedRegion = regions[dataIndex];

                        let studentGroup;
                        if (datasetIndex === 0) {
                            studentGroup = 'Teaching Practice';
                        } else if (datasetIndex === 1) {
                            studentGroup = 'Industrial Internship';
                        } else {
                            studentGroup = 'all';  // Total line means all students
                        }

                        showStudentModal(selectedRegion, studentGroup);
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });

        // Doughnut chart for Number of Students by Group
        const ctx2 = document.getElementById('groupDistributionChart').getContext('2d');
        const groupDistributionChart = new Chart(ctx2, {
            type: 'doughnut',
            data: {
                labels: groups,
                datasets: [{
                    data: groupCounts,
                    backgroundColor: ['#4084f3', '#32aa53'],
                    hoverOffset: 4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false
            }
        });

        // Function to show modal and load student data
        function showStudentModal(region, studentGroup) {
            document.getElementById('region-name').innerText = region;

            fetch(`fetch_students.php?region=${region}&student_group=${studentGroup}`)
                .then(response => response.json())
                .then(data => {
                    const tbody = document.getElementById('student-table-body');
                    tbody.innerHTML = '';  // Clear previous data

                    if (data.length > 0) {
                        data.forEach(student => {
                            const row = `<tr><td>${student.first_name} ${student.surname}</td><td>${student.district}</td></tr>`;
                            tbody.innerHTML += row;
                        });
                    } else {
                        tbody.innerHTML = '<tr><td colspan="2">No students found for this region.</td></tr>';
                    }

                    const studentModal = new bootstrap.Modal(document.getElementById('studentModal'));
                    studentModal.show();
                })
                .catch(error => console.error('Error fetching students:', error));
        }
    </script>
</body>
</html>
